% Rising Air Bubble Technique (v2.1)
% 01/07/2011
%
% Author: Koen Hilgersom (2011)
%
% This script processes images taken during Rising Bubble Technique
% experiments and calculates the area enclosed by the nozzle line and the
% envelope of surfacing air bubbles. In this version, these locations
% should be entered manually by the user.
%
% Outline of the processes:
% 1. Set up the base grid for the transformation based on the measured
%     relative distances of the reference points;
% 2. Load the image (the location on the disk and number should be defined
%     in lines 37-40 of the script before running)
% 3. Mark the bubbles and the nozzle line
% 4. Undistort the image (in case the image was processed before, the saved
%     version of the undistorted image will be loaded)
% 5. Warp the image to the base grid after marking the reference points in
%     the image
% 6. Interpolate the surfacing bubble points (with power function, logarithm
%     and linearly)
%
%---------------------------------------------------------------------------  
%      Copyright (C) 2011 Technische Universiteit Delft, 
%          Koen Hilgersom
%          K.P.Hilgersom@tudelft.nl (correspondence)
% 
%---------------------------------------------------------------------------  

%% start
clc;clear all;close all;

% input data
tmap_folder     =   'D:\kphilgersom\My Documents\RABT'; % folder with transformation matrices for the given camera lens and focal length
folder          =   'D:\kphilgersom\Desktop\Bubble method test\Zoutkamp'; % project folder (folder with the pictures in case subfolder '1 pics'
                                                                           % has not yet been created, otherwise it is the general project folder)
nrPic           =   '0054';  % picture number (written as a 4-digit string, so e.g. '0002' or '0051', etc.)

% determination of the user's screen size
scrsize         =   get(0,'ScreenSize');
scrsize(1)      =   scrsize(1)+20;        scrsize(2)      =   scrsize(2)+6; 
scrsize(3)      =   scrsize(3)-40;        scrsize(4)      =   scrsize(4)-80;
                
%% create / load project and according base grid
base            =   RABT_Base(scrsize,folder);

%% load picture

% check for saved struct of picture variables
picvar  =   exist([folder '\1 pics\picture ' nrPic '.mat'],'file');

switch picvar
    case 0
        % create struct 'RABT' for the specific picture
        RABT.nr         =   nrPic;
    case 2
        % load struct 'RABT' for the specific picture
        load([folder '\1 pics\picture ' nrPic '.mat']);
end

% load picture
Pic             =   imread([folder '\1 pics\DSC_' RABT.nr '.JPG']);
% orientation of picture
if length(Pic(1,:,1))<length(Pic(:,1,1))
    RABT.orientation='portrait';
else
    RABT.orientation='landscape';
end
        
clear nrPic;

%% mark bubbles and the endings of the nozzle line (at both sides at the shore)
[Loc] = RABT_Setpoints(Pic,base.Ninp,scrsize,base.b); % struct 'Loc' stores the coordinates for the different grids

%% undistort

% check for saved version of the undistorted picture
undistort  =   exist([folder '\2 undistorted\' RABT.nr '.tif'],'file'); 

% undistort and save picture (case 0) or load undistorted picture (case 2)
switch undistort
    case 0 % undistort based on the focal length
        
        % get camera name, lens and focal length from metadata and check if a transformation
        %   matrix for undistortion needs to be created
        [RABT,tmapav]    =   RABT_TmapUndistort(RABT,tmap_folder,folder);
        
        % determination of k1-parameter needed to correct for radial distortion
        load distfact % lookup table with distortion factors for Nikkor AF-S 1855mm f/3.55.6G ED DX lens (saved as struct)
                      %     (loading of the distortion factors should still be made general for a variety of camera lenses)
        ind=find(dist.fl<=RABT.fl,1,'last'); % find the highest focal length in the lookup table that is lower than the actual focal length
        if ind<length(dist.fl)
            RABT.k1=dist.df(ind)*(dist.fl(ind+1)-RABT.fl)/(dist.fl(ind+1)-dist.fl(ind))+...
                dist.df(ind+1)*(RABT.fl-dist.fl(ind))/(dist.fl(ind+1)-dist.fl(ind)); 
        elseif RABT.fl==55
            RABT.k1=dist.df(end);
        else
            error('Distortion factor unknown for this focal length');
        end

        % correction for barrel distortion
        [Pic2,tmap] = RABT_Fundistort(Pic,RABT.fl,RABT.k1,[tmap_folder '\' RABT.cn '\' RABT.ln2],tmapav,RABT.orientation); 
        
        % save image and the transformation matrix used for the undistort process
        imwrite(Pic2,[folder '\2 undistorted\' RABT.nr '.tif'],'tif');
        save([folder '\1 pics\picture ' RABT.nr],'RABT');
        if tmapav==0
            save([tmap_folder '\' RABT.cn '\' RABT.ln2 '\' num2str(RABT.fl) 'mm_' RABT.orientation],'tmap');
        end
        
    case 2 % load undistorted image and the according transformation matrix
        Pic2=imread([folder '\2 undistorted\' RABT.nr '.tif']);
        
        % create and save or load transformation matrix for this camera, lens and focal length
        tmapav = exist([tmap_folder '\' RABT.cn '\' RABT.ln2 '\' num2str(RABT.fl) 'mm_' RABT.orientation '.mat'],'file');
        if tmapav==0
            [~,tmap] = RABT_Fundistort(NaN*Pic2,RABT.fl,RABT.k1,[tmap_folder '\' RABT.cn '\' RABT.ln2],tmapav,RABT.orientation);
            save([tmap_folder '\' RABT.cn '\' RABT.ln2 '\' num2str(RABT.fl) 'mm_' RABT.orientation],'tmap');
        else
            load([tmap_folder '\' RABT.cn '\' RABT.ln2 '\' num2str(RABT.fl) 'mm_' RABT.orientation '.mat']);
        end
        
        disp(   ['The corrected picture is loaded from ' folder '\2 undistorted\' RABT.nr '.tif']);
end

% transform coordinates of the bubbles and the nozzle line towards the base grid coordinates
for k=1:base.Ninp
    [minp,loc1]=min((tmap(:,:,1)-Loc.bubbleOrig(k,1)*ones(size(tmap(:,:,1)))).^2+(tmap(:,:,2)-Loc.bubbleOrig(k,2)*ones(size(tmap(:,:,1)))).^2);
    [~,loc2]=min(minp);
    Loc.bubbleUnd(k,:)=[loc2,loc1(loc2)];
end
for k=1:2
    [minp,loc1]=min((tmap(:,:,1)-Loc.SidePOrig(k,1)*ones(size(tmap(:,:,1)))).^2+(tmap(:,:,2)-Loc.SidePOrig(k,2)*ones(size(tmap(:,:,1)))).^2);
    [~,loc2]=min(minp);
    Loc.SidePUnd(k,:)=[loc2,loc1(loc2)];
end

% show undistorted image
figure(2); clf;
imshow(Pic2);

clear Pic

%% morph picture

% load the base image towards which your image will be warped
baseread=imread([folder '\base grid.tif']);
baseread2=imresize(baseread,base.fact/base.scale,'Method','bilinear'); %in this step, the base image is scaled
clear baseread

% set up the transformation matrix based on the locations of the reference
% points in the original image and in the base grid
base.Xd=ones(length(base.X),1)*((base.dp(1)-1)*base.fact/base.scale-base.da(1));
base.Yd=ones(length(base.Y),1)*(size(baseread2,1)-(base.dp(2)-1)*base.fact/base.scale+base.da(3)/base.scale);
[in_points, out_points]=cpselect(Pic2,baseread2,[800,800;1000,800;800,1000;1000,1000],[round(base.Xd+base.X*1000/base.scale) round(base.Yd-base.Y*1000/base.scale)],'Wait',true);
TF=maketform('projective',in_points,out_points);

% warp the coordinates of the points of interest to the base grid
[Loc.bubbleTF]=tformfwd(TF,Loc.bubbleUnd(:,1),Loc.bubbleUnd(:,2));
[Loc.SidePTF]=tformfwd(TF,Loc.SidePUnd(:,1),Loc.SidePUnd(:,2));

% warp the complete picture (in next versions, this will be optional; it will
% then be recommended to only warp the points of interest and warp the
% calculated area back to the original image, since this is a major save in time)
[PicT,XDATA,YDATA]=imtransform(Pic2, TF,'XData',[min([out_points(:,1);Loc.bubbleTF(:,1);Loc.SidePTF(:,1)])...
    max([out_points(:,1);Loc.bubbleTF(:,1);Loc.SidePTF(:,1)])],'YData',[min([out_points(:,2);Loc.bubbleTF(:,2);Loc.SidePTF(:,2)])...
    max([out_points(:,2);Loc.bubbleTF(:,2);Loc.SidePTF(:,2)])],'XYScale',1);

% location of the bubbles in the newly defined region of in terest of the picture
Loc.bubbleTF(:,1)=Loc.bubbleTF(:,1)-XDATA(1)*ones(length(Loc.bubbleTF(:,1)),1);
Loc.bubbleTF(:,2)=Loc.bubbleTF(:,2)-YDATA(1)*ones(length(Loc.bubbleTF(:,2)),1);
Loc.SidePTF(:,1)=Loc.SidePTF(:,1)-XDATA(1)*ones(length(Loc.SidePTF(:,1)),1);
Loc.SidePTF(:,2)=Loc.SidePTF(:,2)-YDATA(1)*ones(length(Loc.SidePTF(:,2)),1);

clear Pic2 TF

% show transformed image
figure(3)
clf
imshow(PicT)
hold on
scatter(Loc.bubbleTF(:,1),Loc.bubbleTF(:,2),'FaceColor','g');
scatter(Loc.SidePTF(:,1),Loc.SidePTF(:,2),'FaceColor','r');
set(gcf,'Position',scrsize);

%% calculation area

% area linear interpolation (green)
figure(4)
clf
imshow(PicT)
hold on
set(gcf,'Position',scrsize);
scatter(Loc.bubbleTF(:,1),Loc.bubbleTF(:,2),'FaceColor','g');
area([Loc.bubbleTF(:,1);Loc.SidePTF(:,1);Loc.bubbleTF(1,1)],[Loc.bubbleTF(:,2);Loc.SidePTF(:,2);Loc.bubbleTF(1,2)],'FaceColor','g');
AG=polyarea([Loc.bubbleTF(:,1);Loc.SidePTF(:,1);Loc.bubbleTF(1,1)],[Loc.bubbleTF(:,2);Loc.SidePTF(:,2);Loc.bubbleTF(1,2)])/10^6*base.scale^2;
text(base.dq(1),size(PicT,1)+120/base.scale,['Area = ' num2str(AG,3) ' m^2']);

QG=base.vr*AG*10^3; %discharge (L/s)
text(base.dq(1),size(PicT,1)+300/base.scale,['Discharge = ' num2str(QG,3) ' L/s']);

Arr(1)=java.lang.String([folder '\3 result\lin' RABT.nr]);c(1)=cell(Arr);
saveSameSize(gcf, 'format', 'tiff', 'file', c{1});

% area log interpolation (yellow)
figure(5)
clf
imshow(PicT)
hold on
set(gcf,'Position',scrsize);
[vecX,vecY]=logFitHorProfile2(Loc); %obtain log flow profile
scatter(Loc.bubbleTF(:,1),Loc.bubbleTF(:,2),'FaceColor','y');
area([vecX;Loc.SidePTF(:,1);vecX(1)],[vecY;Loc.SidePTF(:,2);vecY(1)],'FaceColor','y');
AY=polyarea([vecX;Loc.SidePTF(:,1);vecX(1)],[vecY;Loc.SidePTF(:,2);vecY(1)])/10^6*base.scale^2;
text(base.dq(1),size(PicT,1)+120/base.scale,['Area = ' num2str(AY,3) ' m^2']);

QY=base.vr*AY*10^3; %discharge (L/s)
text(base.dq(1),size(PicT,1)+300/base.scale,['Discharge = ' num2str(QY,3) ' L/s']);

Arr(1)=java.lang.String([folder '\3 result\log' RABT.nr]);c(1)=cell(Arr);
saveSameSize(gcf, 'format', 'tiff', 'file', c{1});

% area power interpolation (red)
figure(6)
clf
imshow(PicT)
hold on
set(gcf,'Position',scrsize);
[vecX,vecY]=powerFitHorProfile(Loc);
scatter(Loc.bubbleTF(:,1),Loc.bubbleTF(:,2),'FaceColor','r');
area([vecX;Loc.SidePTF(:,1);vecX(1)],[vecY;Loc.SidePTF(:,2);vecY(1)],'FaceColor','r');
AR=polyarea([vecX;Loc.SidePTF(:,1);vecX(1)],[vecY;Loc.SidePTF(:,2);vecY(1)])/10^6*base.scale^2;
text(base.dq(1),size(PicT,1)+120/base.scale,['Area = ' num2str(AR,3) ' m^2']);

QR=base.vr*AR*10^3; %discharge (L/s)
text(base.dq(1),size(PicT,1)+300/base.scale,['Discharge = ' num2str(QR,3) ' L/s']);

Arr(1)=java.lang.String([folder '\3 result\power' RABT.nr]);c(1)=cell(Arr);
saveSameSize(gcf, 'format', 'tiff', 'file', c{1});

% export data to txt-file
export(dataset({AG,'ALin'},{AY, 'ALog'},{AR, 'APower'},{QG,'QLin'},{QY, 'QLog'},...
    {QR, 'QPower'},{[Loc.bubbleTF(:,1);Loc.SidePTF(:,1)]','coorx'},...
    {[Loc.bubbleTF(:,2);Loc.SidePTF(:,2)]', 'coory'}), 'file',...
    [folder '\4 results txt\result picture ' RABT.nr '.txt'],...
    'delimiter', ',', 'WriteVarNames',true);